"""
Archiver dynamic web query handler

Refer to https://trac.sdss3.org/wiki/Ops/Arch/Server for details.
"""

# Created 21-Apr-2009 by David Kirkby (dkirkby@uci.edu)

from builtins import str
from builtins import object
import os,os.path
import time
from operator import itemgetter
from datetime import datetime
from twisted.web import resource,server,static
from twisted.internet import defer

from opscore.utility import html
from archiver import database,actors

class WebError(Exception):
    pass

class SessionState(object):
    pass

def deferredRender(what,request):
    """
    Renders the results of a deferred request
    """
    request.write(str(what))
    request.finish()

class WebQuery(resource.Resource):
    """
    Implements a web resource that handles GET and POST requests.
    
    Requests are matched to a per-service and per-client-window session
    state object. The client must include a 'uid' parameter in the
    request to support the per-client-window session state mapping.
    """
    isLeaf = True
    title = 'SDSS3 Archiver'

    services = [ ]
    
    def __init__(self,serviceName):
        if serviceName in self.services:
            raise WebError('Service already registered: %s' % serviceName)
        self.serviceName = serviceName
        WebQuery.services.append(serviceName)

    def get_arg(self,name,default=None):
        try:
            return self.args[name][0]
        except (AttributeError,KeyError):
            return default

    def prepare(self,request):
        self.args = request.args
        request.sitepath = [self.serviceName]
        session = request.getSession()
        if not 'uid' in request.args:
            uid = 0
        else:
            uid = request.args['uid'][0]
        #print 'Retrieving session state for uid',uid,'in',id(session)
        if not hasattr(session,'state'):
            session.state = { }
        if not uid in session.state:
            session.state[uid] = SessionState()
        return (session,session.state[uid])

    def render_GET(self, request):
        (session,state) = self.prepare(request)
        doc = html.HTMLDocument(
            html.Head(title=self.title,icon='/static/favicon.ico'),
            html.Body(
                html.Div('sdss3archiver',html.Span(id='menu'),id='header'),
                html.Div(id='content')
            )
        )
        doc.head.css = [
            '/static/jquery-ui-1.7.2.custom.css',
            '/static/styles.css']
        doc.head.js = [
            '/static/jquery-1.3.2.min.js',
            '/static/jquery-ui-1.7.2.custom.min.js',
            '/static/jquery.flot.pack.js']
        # IE does not yet support the canvas tag that flot needs so the
        # following HTML hack only loads the excanvas library on IE
        doc.head.raw = [
            '<!--[if IE]><script language="javascript" type="text/javascript" ' +
            'src="/static/excanvas.pack.js"></script><![endif]-->'
        ]
        for separator,service in enumerate(WebQuery.services):
            if separator:
                doc['menu'].append(' | ')
            if service == self.serviceName:
                doc['menu'].append(service)
            else:
                doc['menu'].append(html.A(service,href='/archiver/%s' % service))
        self.GET(request,session,state,doc['content'])
        return str(doc)
            
    def render_POST(self, request):
        (session,state) = self.prepare(request)
        result = self.POST(request,session,state)
        if isinstance(result,defer.Deferred):
            # the result is being generated asynchronously
            result.addCallback(deferredRender,request)
            return server.NOT_DONE_YET
        else:
            # we already have the result so return it now
            return str(result) 

class FilteredSite(server.Site):
    """
    A web site that filters out request logs from certain sites.

    Use this class to filter out the log messages that would be
    generated by, for example, periodic AJAX requests. Set a
    filtered_paths attribute of your site instance to a list of URI
    regular-expression patterns that should be filtered use re.match.
    """
    def log(self,request):
        if hasattr(self,'filtered_paths'):
            for path in self.filtered_paths:
                if path.match(request.uri):
                    return None
        return server.Site.log(self,request)

class ArchiverSite(FilteredSite):
    """
    The archiver web site
    """
    def __init__(self,info,options):
        # create an empty root for our site
        root = resource.Resource()
        # serve our static content
        root.putChild('static',static.File(os.path.join(info.productDir,'web')))
        # serve our dynamic content
        dynamic = resource.Resource()
        root.putChild('archiver',dynamic)
        import archiver.browse
        dynamic.putChild('browse',archiver.browse.BrowseHandler('browse',options))
        dynamic.putChild('info',InfoHandler('info',info,options))
        #dynamic.putChild('log',LogHandler('log'))
        # create the site we will serve
        FilteredSite.__init__(self,root)

class InfoHandler(WebQuery):

    title = 'SDSS3 Archiver Information'

    def __init__(self,serviceName,info,options):
        self.info = info
        self.options = options
        WebQuery.__init__(self,serviceName)

    def GET(self,request,session,state,content):
        
        now = datetime.now()
        content.append('Report created %s' % now)
        
        content.append(html.H1('Server Status'))
        info = self.info
        elapsed = now - datetime.fromtimestamp(info.startedAt)
        if database.Table.lastActivity:
            last = ('Last table activity %s ago (%d active tables)' %
                (now - datetime.fromtimestamp(database.Table.lastActivity),
                len(database.Table.registry)))
        else:
            last = 'No table activity yet'
        status = html.Ul(
            html.Li(last),
            html.Li('Running since %s (%s ago)' % (time.ctime(info.startedAt),elapsed)),
            html.Li('Started by %s using %s' % (info.user,info.commandLine)),
            html.Li('Running as PID %d on %s' % (info.pid,info.host)),
            html.Li('Product directory is %s' % info.productDir)
        )
        content.append(status)

        content.append(html.H1('Runtime Configuration'))
        optionsTable = html.Table()
        header = html.Tr()
        header.extend([html.Th(hdr) for
            hdr in ('option','value','default','description')])
        optionsTable.append(header)
        for rowData in self.options._summary_table:
            optionRow = html.Tr()
            optionRow.extend([html.Td(field) for field in rowData])
            optionsTable.append(optionRow)
        content.append(optionsTable)

        content.append(html.H1('Actors'))
        actorsTable = html.Table()
        header = html.Tr()
        header.extend([html.Th(hdr) for hdr in
            ('name','version','#keywords','#active','#updates','most active keywords')])
        actorsTable.append(header)
        for name in sorted(actors.Actor.registry):
            actor = actors.Actor.registry[name]
            actorRow = html.Tr()
            actorRow.append(html.Td(name))
            if actor.kdict:
                actorRow.append(html.Td('%d.%d' % actor.kdict.version))
                actorRow.append(html.Td(len(actor.kdict.keys)))
            else:
                actorRow.append(html.Td('no dictionary found'))
                actorRow.append(html.Td('?'))
            nKeys = 0
            nUpdates = 0
            mostActive = ''
            for key,count in sorted(iter(actor.keyStats.items()),
                key=itemgetter(1),reverse=True):
                if nKeys < 10:
                    mostActive += '%s (%d) ' % (key,count)
                nKeys += 1
                nUpdates += count
            actorRow.append(html.Td(nKeys))
            actorRow.append(html.Td(nUpdates))
            actorRow.append(html.Td(mostActive))
            actorsTable.append(actorRow)
        content.append(actorsTable)

    def POST(self,request,session,state):
        pass

class LogHandler(WebQuery):

    title = 'SDSS3 Archiver Log'

    def GET(self,request,session,state,content):
        now = datetime.now()
        content.append('Log generated %s' % now)
        content.append(html.H1('Recent Raw Messages'))
        try:
            table = database.Table.attach('reply_raw')
        except database.DatabaseException:
            content.append('No messages have been logged yet')
            return
        logTable = html.Table()
        header = html.Tr()
        header.extend([html.Th(hdr) for hdr in ('id','tai','message')])
        logTable.append(header)
        for (idnum,tai,msg) in table.rowBuffer[::-1]:
            logRow = html.Tr()
            logRow.extend([html.Td(field) for field in (idnum,tai,msg)])
            logTable.append(logRow)
        content.append(logTable)

    def POST(self,request,session,state):
        pass
